/*
 * can_tx_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * vCANTask is called by main() to configure the CAN0 module for transmitting
 * data at a 500kHz bit rate and enables interrupts.  Afterwards, a total of
 * five tasks and one queue are created.
 *
 * The first two tasks, prvTransmit1Task and prvTransmit2Task, are mapped to
 * message object 1 and message object 2.  A changing test pattern is sent to
 * each message object at a random interval.  Once the data is written to the
 * message object, it is sent on the physical CAN bus.
 *
 * The next two tasks, prvTransmit3Task and prvTransmit4Task, share message
 * object 3.  These two tasks do not write to the message object 3 directly but
 * rather send the data through a queue.
 *
 * The final task, prvQueueReceiveTask1, receives the data from the queue and
 * then writes to message object 3.  This is to prevent the prvTransmit3Task
 * and prvTransmit4Task tasks from writing to the message object 3
 * asynchronously.  The queue serves as a mutex or a gatekeeper.
 *
 * xCANHandler is the hardware CAN interrupt handler which is used to both
 * confirm message transmission as well as keep count of how many messages have
 * been sent.
 *
 * This example uses the following peripherals and I/O signals on the
 * EK-TM4C1294XL:
 * - CAN0RX - PA0
 * - CAN0TX - PA1
 *
 * !!!- TODO -!!!
 * On the EK-TM4C1294XL LaunchPad, the jumper JP4 and JP5 must be changed to
 * the vertical position in order to bring CAN0RX and CAN0TX signals to the
 * BoosterPack 2 header.  In its default horizontal positions, UART0 is used
 * for the ICDI virtual UART and CAN is not present on the BoosterPack headers.
 * When the jumpers are configured for CAN on the BoosterPack, then UART2
 * must be used for the ICDI virtual UART.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "inc/hw_can.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/can.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "drivers/rtos_hw_drivers.h"
#include "utils/uartstdio.h"
#include "uart_task.h"
/*-----------------------------------------------------------*/

/*
 * The number of items the queue can hold.  This is 4 as the receive task
 * will remove items as they are added, meaning the send task should always
 * find the queue empty.
 */
#define mainQUEUE_LENGTH                    ( 4 )

/*
 * Queue used to send and receive complete struct AMessage structures.
 */
QueueHandle_t xStructQueue = NULL;

/*
 * A counter that keeps track of the number of times the TX interrupt has
 * occurred, which should match the number of TX messages that were sent.
 */
volatile uint32_t g_ui32MsgCount = 0;

/*
 * A flag to indicate that some transmission error occurred.
 */
volatile bool g_bErrFlag = 0;

/*
 * The notification used by the task.
 */
TaskHandle_t xTask3CANHandle = NULL;
TaskHandle_t xTask4CANHandle = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvTransmit1Task( void *pvParameters );
static void prvTransmit2Task( void *pvParameters );
static void prvTransmit3Task( void *pvParameters );
static void prvTransmit4Task( void *pvParameters );
static void prvQueueReceiveTask1 ( void *pvParameters );

/*
 * Called by main() to create the CAN application.
 */
void vCANTask( void );

/*
 * This function prints some information about the CAN message to the
 * serial port for information purposes only.
 */
void PrintCANMessageInfo(tCANMsgObject *psCANMsg, uint32_t ui32MsgObj);

/*
 * Hardware configuration for the CAN peripheral.
 */
static void prvConfigureCAN( void );
/*-----------------------------------------------------------*/

void vCANTask( void )
{
    /* Configure the CAN 0 peripheral. */
    prvConfigureCAN();

    /* Create the queue used to send complete struct AMessage structures.  This
     * can also be created after the schedule starts, but care must be task to
     * ensure nothing uses the queue until after it has been created. */
    xStructQueue = xQueueCreate(
                          /* The number of items the queue can hold. */
                          mainQUEUE_LENGTH,
                          /* Size of each item is big enough to hold the
                          whole structure. */
                          sizeof( tCANMsgObject ) );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name queue RX task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - No parameter passed to the task
     *  - The priority assigned to the task.
     *  - The task handle is NULL */
    xTaskCreate( prvQueueReceiveTask1,
                "Queue RX",
                configMINIMAL_STACK_SIZE,
                NULL,
                tskIDLE_PRIORITY + 1,
                NULL );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for Message 1 task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvTransmit1Task,
                 "MsgObj1",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 NULL );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for Message 2 task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvTransmit2Task,
                 "MsgObj2",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 NULL );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for Message 3 task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvTransmit3Task,
                 "MsgObj3", /* prvTransmit3Task and prvTransmit4Task will use
                               the same CAN Message Object 3 on the hardware */
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xTask3CANHandle );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name for Message 4 task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - The parameter passed to the task - just to check the functionality.
     *  - The priority assigned to the task.
     *  - The task handle used for the task notify. */
    xTaskCreate( prvTransmit4Task,
                 "MsgObj3", /* Although it is a different task, the text
                               name is the same as prvTransmit3Task */
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xTask4CANHandle );
}
/*-----------------------------------------------------------*/

static void prvTransmit1Task( void *pvParameters )
{
tCANMsgObject sCANMsgObject1;
uint8_t pui8Msg1[4] = { 0, 0, 0, 0 };

    /* Initialize message object 1 to be able to send CAN message 1.  This
     * message object is not shared so it only needs to be initialized one
     * time, and can be used for repeatedly sending the same message ID. */
    sCANMsgObject1.ui32MsgID = 0x1001;
    sCANMsgObject1.ui32MsgIDMask = 0;
    sCANMsgObject1.ui32Flags = MSG_OBJ_TX_INT_ENABLE;
    sCANMsgObject1.ui32MsgLen = sizeof(pui8Msg1);
    sCANMsgObject1.pui8MsgData = pui8Msg1;

    for (;;)
    {
        /* Send message 1 using CAN controller message object 1. */
        PrintCANMessageInfo(&sCANMsgObject1, 1);
        CANMessageSet(CAN0_BASE, 1, &sCANMsgObject1, MSG_OBJ_TYPE_TX);

        /* Increment the data to be sent. */
        (*(uint32_t *)pui8Msg1)++;

        /* Randomize the time (from 10 to 265) to send the next message. */
        vTaskDelay( ( rand() % 0xFF  + 10) );
    }
}
/*-----------------------------------------------------------*/

static void prvTransmit2Task( void *pvParameters )
{
tCANMsgObject sCANMsgObject2;
uint8_t pui8Msg2[5] = { 2, 2, 2, 2, 2 };

    /* Initialize message object 2 to be able to send CAN message 2.  This
     * message object is not shared so it only needs to be initialized one
     * time, and can be used for repeatedly sending the same message ID. */
    sCANMsgObject2.ui32MsgID = 0x2001;
    sCANMsgObject2.ui32MsgIDMask = 0;
    sCANMsgObject2.ui32MsgLen = sizeof(pui8Msg2);
    sCANMsgObject2.pui8MsgData = pui8Msg2;

    for (;;)
    {
        /* Send message 2 using CAN controller message object 2. */
        PrintCANMessageInfo(&sCANMsgObject2, 2);
        CANMessageSet(CAN0_BASE, 2, &sCANMsgObject2, MSG_OBJ_TYPE_TX);

        /* Increment the data to be sent. */
        (*(uint32_t *)pui8Msg2)++;

        /* Randomize the time (from 10 to 265) to send the next message. */
        vTaskDelay( ( rand() % 0xFF  + 10) );
    }
}
/*-----------------------------------------------------------*/

static void prvTransmit3Task( void *pvParameters )
{
tCANMsgObject sCANMsgObject3;
uint8_t pui8Msg3[6] = { 3, 3, 3, 3, 3, 3 };

    /* Initialize message object 3 to be able to send CAN message 3.  This
     * message object is not shared so it only needs to be initialized one
     * time, and can be used for repeatedly sending the same message ID. */
    sCANMsgObject3.ui32MsgID = 0x3001;
    sCANMsgObject3.ui32MsgIDMask = 0;
    sCANMsgObject3.ui32MsgLen = sizeof(pui8Msg3);
    sCANMsgObject3.pui8MsgData = pui8Msg3;

    for (;;)
    {
        /* Send the entire structure by value to the queue. */
        xQueueSend( /* The handle of the queue. */
                    xStructQueue,
                    /* The address of the sCANMsgObject3 variable.
                     * sizeof( struct sCANMsgObject3 ) bytes are copied from
                     * here into the queue. */
                    ( void * ) &sCANMsgObject3,
                    /* Block time of 0 says don't block if the queue is already
                     * full.  Check the value returned by xQueueSend() to know
                     * if the message was sent to the queue successfully. */
                    ( TickType_t ) 0 );

        /* Increment the data to be sent. */
        (*(uint32_t *)pui8Msg3)++;

        /* Randomize the time (from 10 to 265) to send the next message */
        vTaskDelay( ( rand() % 0xFF  + 10) );
    }
}
/*-----------------------------------------------------------*/

static void prvTransmit4Task( void *pvParameters )
{
tCANMsgObject sCANMsgObject3;
uint8_t pui8Msg4[8] = { 4, 4, 4, 4, 5, 5, 5, 5 };

    /* Initialize message object 4 to be able to send CAN message 4.  This
     * message object is not shared so it only needs to be initialized one
     * time, and can be used for repeatedly sending the same message ID. */
    sCANMsgObject3.ui32MsgID = 0x3002;
    sCANMsgObject3.ui32MsgIDMask = 0;
    sCANMsgObject3.ui32MsgLen = sizeof(pui8Msg4);
    sCANMsgObject3.pui8MsgData = pui8Msg4;

    for (;;)
    {
        /* Send the entire structure by value to the queue. */
        xQueueSend( /* The handle of the queue. */
                    xStructQueue,
                    /* The address of the sCANMsgObject3 variable.
                     * sizeof( struct sCANMsgObject3 ) bytes are copied from
                     * here into the queue. */
                    ( void * ) &sCANMsgObject3,
                    /* Block time of 0 says don't block if the queue is already
                     * full.  Check the value returned by xQueueSend() to know
                     * if the message was sent to the queue successfully. */
                    ( TickType_t ) 0 );

        /* Increment the data to be sent. */
        (*(uint32_t *)&pui8Msg4[0])++;
        (*(uint32_t *)&pui8Msg4[4])--;

        /* Randomize the time (from 10 to 265) to send the next message. */
        vTaskDelay( ( rand() % 0xFF  + 10) );
    }
}
/*-----------------------------------------------------------*/

static void prvQueueReceiveTask1( void *pvParameters )
{
tCANMsgObject xRxedStructure;

    for (;;)
    {
        if( xStructQueue != NULL )
        {
            /* Receive a message from the created queue to hold complex struct
             * AMessage structure.  Block for 10 ticks if a message is not
             * immediately available.  The value is read into a struct xRxedStructure
             * variable, so after calling xQueueReceive() xRxedStructure will
             * hold a copy of xMessage. */
            if( xQueueReceive( xStructQueue,
                               &( xRxedStructure ),
                               ( TickType_t ) 10 ) == pdPASS )
            {
                /* Send the received message structure to CAN Msg Object 3. */
                PrintCANMessageInfo(&xRxedStructure, 3);
                CANMessageSet(CAN0_BASE, 3, &xRxedStructure, MSG_OBJ_TYPE_TX);
            }
        }
    }
}
/*-----------------------------------------------------------*/

void
PrintCANMessageInfo(tCANMsgObject *psCANMsg, uint32_t ui32MsgObj)
{
unsigned int uIdx;

    /* This function prints some information about the CAN message to the
     * serial port for information purposes only. */
    prvUARTPrintf("Sending msg: obj=%d ID=0x%04X msg=0x", ui32MsgObj,
               psCANMsg->ui32MsgID);

    for(uIdx = 0; uIdx < psCANMsg->ui32MsgLen; uIdx++)
    {
        prvUARTPrintf("%02X ", psCANMsg->pui8MsgData[uIdx]);
    }

    prvUARTPrintf("\n");
}
/*-----------------------------------------------------------*/

static void prvConfigureCAN( void )
{
    /* Configure the GPIO pin muxing to select CAN0 functions for these pins.
     * This step selects which alternate function is available for these pins.
     * This is necessary if your part supports GPIO pin function muxing.
     * Consult the data sheet to see which functions are allocated per pin.
     * TODO: change this to select the port/pin you are using. */
    GPIOPinConfigure(GPIO_PA0_CAN0RX);
    GPIOPinConfigure(GPIO_PA1_CAN0TX);

    /* Enable the alternate function on the GPIO pins.  The above step selects
     * which alternate function is available.  This step actually enables the
     * alternate function instead of GPIO for these pins.
     * TODO: change this to match the port/pin you are using. */
    GPIOPinTypeCAN(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    /* The GPIO port and pins have been set up for CAN.  The CAN peripheral
     * must be enabled. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CAN0);

    /* Initialize the CAN controller. */
    CANInit(CAN0_BASE);

    /* Set up the bit rate for the CAN bus.  This function sets up the CAN
     * bus timing for a nominal configuration.  You can achieve more control
     * over the CAN bus timing by using the function CANBitTimingSet() instead
     * of this one, if needed. In this example, the CAN bus is set to 500 kHz.
     */
    CANBitRateSet(CAN0_BASE, configCPU_CLOCK_HZ, 500000);

    /* Enable interrupts on the CAN peripheral.  This example uses static
     * allocation of interrupt handlers which means the name of the handler
     * is in the vector table of startup code.  If you want to use dynamic
     * allocation of the vector table, then you must also call CANIntRegister()
     * here. */
    CANIntEnable(CAN0_BASE, CAN_INT_MASTER | CAN_INT_ERROR | CAN_INT_STATUS);

    /* Enable the CAN interrupt on the processor (NVIC). */
    IntEnable(INT_CAN0);

    /* Enable the CAN for operation. */
    CANEnable(CAN0_BASE);
}
/*-----------------------------------------------------------*/

void
xCANHandler(void)
{
uint32_t ui32Status;

    /* Read the CAN interrupt status to find the cause of the interrupt. */
    ui32Status = CANIntStatus(CAN0_BASE, CAN_INT_STS_CAUSE);

    /* If the cause is a controller status interrupt, then get the status. */
    if(ui32Status == CAN_INT_INTID_STATUS)
    {
        /* Read the controller status.  This will return a field of status
         * error bits that can indicate various errors.  Error processing
         * is not done in this example for simplicity.  Refer to the
         * API documentation for details about the error status bits.
         * The act of reading this status will clear the interrupt.  If the
         * CAN peripheral is not connected to a CAN bus with other CAN devices
         * present, then errors will occur and will be indicated in the
         * controller status. */
        ui32Status = CANStatusGet(CAN0_BASE, CAN_STS_CONTROL);

        /* Set a flag to indicate some errors may have occurred. */
        g_bErrFlag = 1;
    }

    /* Check if the cause is message object 1, which what we are using for
     * sending messages. */
    else if(ui32Status == 1)
    {
        /* Getting to this point means that the TX interrupt occurred on
         * message object 1, and the message TX is complete.  Clear the
         * message object interrupt. */
        CANIntClear(CAN0_BASE, 1);

        /* Increment a counter to keep track of how many messages have been
         * sent.  In a real application this could be used to set flags to
         * indicate when a message is sent. */
        g_ui32MsgCount++;

        /* Since the message was sent, clear any error flags. */
        g_bErrFlag = 0;
    }

    /* Otherwise, something unexpected caused the interrupt.  This should
     * never happen. */
    else
    {
        /* Spurious interrupt handling can go here. */
    }
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}

